<?php

namespace App\Controller;

use App\Entity\Country;
use App\Repository\CountryRepository;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Serializer\Normalizer\AbstractNormalizer;
use Symfony\Component\Serializer\SerializerInterface;
use Symfony\Component\Validator\Validator\ValidatorInterface;

final class CountryController extends AbstractController
{
    private $em;
    private $validator;
    private $serializer;

    public function __construct(EntityManagerInterface $em, ValidatorInterface $validator, SerializerInterface $serializer)
    {
        $this->em = $em;
        $this->validator = $validator;
        $this->serializer = $serializer;
    }

    # Render template countries - paises
    #[Route('/paises', name: 'countries.index', methods:['GET'])]   
    public function index(): Response
    {
        return $this->render('country/index.html.twig');
    }

    # GET /api/countries
    #[Route('/api/countries', name: 'api.countries.list', methods: ['GET'])]
    public function list(Request $request, CountryRepository $repo): JsonResponse
    {
        $nombre = $request->query->get('nombre');
        $countries = ($nombre) ? $repo->findByNombre($nombre) : $repo->findAll();

        return $this->json(
            data: $countries,
            status: Response::HTTP_OK,
            context: [AbstractNormalizer::GROUPS => ['country:read']]
        );
    }

    # GET /api/countries/:id
    #[Route('/api/countries/{id}', name: 'api.countries.show', methods: ['GET'])]
    public function show(Country $country): JsonResponse
    {
        return $this->json(
            data: $country,
            status: Response::HTTP_OK,
            context: [AbstractNormalizer::GROUPS => ['country:read']]
        );
    }

    # POST /api/countries
    #[Route('/api/countries', name: 'api.countries.create', methods: ['POST'])]  
    public function create(Request $request): JsonResponse
    {
        $data = $request->getContent();
        $country = $this->serializer->deserialize($data, Country::class, 'json');
        $this->em->persist($country);
        $this->em->flush();

        return $this->json(
            data: $country,
            status: Response::HTTP_CREATED,
            context: [AbstractNormalizer::GROUPS => ['country:read']]
        );
    }

    # PUT /api/countries/:id
    #[Route('/api/countries/{id}', name: 'api.countries.update', methods: ['PUT'])]
    public function update(Request $request, Country $country): JsonResponse
    {
        if (!$country) return $this->json([]); # por si no existe

        $data = $request->getContent();
        $this->serializer->deserialize($data, Country::class, 'json', [AbstractNormalizer::OBJECT_TO_POPULATE => $country]);
        $this->em->flush();

        return $this->json(
            data: $country,
            status: Response::HTTP_OK,
            context: [AbstractNormalizer::GROUPS => ['country:read']]
        );
    }
}
