<?php

namespace App\Controller;

use App\Entity\Hotel;
use App\Repository\HotelRepository;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Serializer\Normalizer\AbstractNormalizer;
use Symfony\Component\Serializer\SerializerInterface;
use Symfony\Component\Validator\Validator\ValidatorInterface;

final class HotelController extends AbstractController
{
    private $em;
    private $validator;
    private $serializer;

    public function __construct(EntityManagerInterface $em, ValidatorInterface $validator, SerializerInterface $serializer)
    {
        $this->em = $em;
        $this->validator = $validator;
        $this->serializer = $serializer;
    }

    # Render template hotel - hoteles
    #[Route('/hoteles', name: 'hotels.index', methods:['GET'])]   
    public function index(): Response
    {
        return $this->render('hotel/index.html.twig');
    }

    # GET /api/hotels
    #[Route('/api/hotels', name: 'api.hotels.list', methods: ['GET'])]
    public function list(Request $request, HotelRepository $repo): JsonResponse
    {
        $nombre = $request->query->get('nombre');
        $hotels = ($nombre) ? $repo->findByNombre($nombre) : $repo->findAll();

        return $this->json(
            data: $hotels,
            status: Response::HTTP_OK,
            context: [AbstractNormalizer::GROUPS => ['hotel:read']] 
        );
    }

    # GET /api/hotels/:id
    #[Route('/api/hotels/{id}', name: 'api.hotels.show', methods: ['GET'])]
    public function show(Hotel $hotel): JsonResponse
    {
        return $this->json(
            data: $hotel,
            status: Response::HTTP_OK,
            context: [AbstractNormalizer::GROUPS => ['hotel:read']]
        );
    }

    # POST /api/hotels
    #[Route('/api/hotels', name: 'api.hotels.create', methods: ['POST'])]  
    public function create(Request $request): JsonResponse
    {
        $data = $request->getContent();
        $hotel = $this->serializer->deserialize($data, Hotel::class, 'json');
        $this->em->persist($hotel);
        $this->em->flush();

        return $this->json(
            data: $hotel,
            status: Response::HTTP_CREATED,
            context: [AbstractNormalizer::GROUPS => ['hotel:read']]
        );
    }

    # PUT /api/hotels/:id
    #[Route('/api/hotels/{id}', name: 'api.hotels.update', methods: ['PUT'])]
    public function update(Request $request, Hotel $hotel): JsonResponse
    {
        if (!$hotel) return $this->json([]); # por si no existe

        $data = $request->getContent();
        $this->serializer->deserialize($data, Hotel::class, 'json', [AbstractNormalizer::OBJECT_TO_POPULATE => $hotel]);
        $this->em->flush();

        return $this->json(
            data: $hotel,
            status: Response::HTTP_OK,
            context: [AbstractNormalizer::GROUPS => ['hotel:read']]
        );
    }
}
