<?php

namespace App\Controller;

use App\Entity\PaymentMethod;
use App\Repository\PaymentMethodRepository;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Serializer\Normalizer\AbstractNormalizer;
use Symfony\Component\Serializer\SerializerInterface;
use Symfony\Component\Validator\Validator\ValidatorInterface;

final class PaymentMethodController extends AbstractController
{
    private $em;
    private $validator;
    private $serializer;

    public function __construct(EntityManagerInterface $em, ValidatorInterface $validator, SerializerInterface $serializer)
    {
        $this->em = $em;
        $this->validator = $validator;
        $this->serializer = $serializer;
    }

    # Render template payment methods - metodos de pago
    #[Route('/metodos-de-pago', name: 'payment_methods.index', methods:['GET'])]
    public function index(): Response
    {
        return $this->render('payment_method/index.html.twig');
    }

    # GET /api/payment_methods
    #[Route('/api/payment_methods', name: 'api.payment_methods.list', methods: ['GET'])]
    public function list(Request $request, PaymentMethodRepository $repo): JsonResponse
    {
        $nombre = $request->query->get('nombre');
        $methods = ($nombre) ? $repo->findByNombre($nombre) : $repo->findAll();

        return $this->json(
            data: $methods,
            status: Response::HTTP_OK,
            context: [AbstractNormalizer::GROUPS => ['paymentMethod:read']]
        );
    }

    # GET /api/payment_methods/:id
    #[Route('/api/payment_methods/{id}', name: 'api.payment_methods.show', methods: ['GET'])]
    public function show(PaymentMethod $paymentMethod): JsonResponse
    {
        return $this->json(
            data: $paymentMethod,
            status: Response::HTTP_OK,
            context: [AbstractNormalizer::GROUPS => ['paymentMethod:read']]
        );
    }

    # POST /api/payment_methods
    #[Route('/api/payment_methods', name: 'api.payment_methods.create', methods: ['POST'])]  
    public function create(Request $request): JsonResponse
    {
        $data = $request->getContent();
        $paymentMethod = $this->serializer->deserialize($data, PaymentMethod::class, 'json');
        $this->em->persist($paymentMethod);
        $this->em->flush();

        return $this->json(
            data: $paymentMethod,
            status: Response::HTTP_CREATED,
            context: [AbstractNormalizer::GROUPS => ['paymentMethod:read']]
        );
    }

    # PUT /api/payment_methods/:id
    #[Route('/api/payment_methods/{id}', name: 'api.payment_methods.update', methods: ['PUT'])]
    public function update(Request $request, PaymentMethod $paymentMethod): JsonResponse
    {
        if (!$paymentMethod) return $this->json([]); # por si no existe

        $data = $request->getContent();
        $this->serializer->deserialize($data, PaymentMethod::class, 'json', [AbstractNormalizer::OBJECT_TO_POPULATE => $paymentMethod]);
        $this->em->flush();

        return $this->json(
            data: $paymentMethod,
            status: Response::HTTP_OK,
            context: [AbstractNormalizer::GROUPS => ['paymentMethod:read']]
        );
    }
}
