<?php

namespace App\Controller;

use App\Entity\Service;
use App\Repository\ServiceRepository;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Serializer\Normalizer\AbstractNormalizer;
use Symfony\Component\Serializer\SerializerInterface;
use Symfony\Component\Validator\Validator\ValidatorInterface;

final class ServiceController extends AbstractController
{
    private $em;
    private $validator;
    private $serializer;

    public function __construct(EntityManagerInterface $em, ValidatorInterface $validator, SerializerInterface $serializer)
    {
        $this->em = $em;
        $this->validator = $validator;
        $this->serializer = $serializer;
    }

    # Render template countries - paises
    #[Route('/servicios', name: 'services.index', methods:['GET'])]    
    public function index(): Response
    {
        return $this->render('service/index.html.twig');
    }

    # GET /api/services
    #[Route('/api/services', name: 'api.services.list', methods: ['GET'])]
    public function list(Request $request, ServiceRepository $repo): JsonResponse
    {
        $nombre = $request->query->get('nombre');
        $countries = ($nombre) ? $repo->findByNombre($nombre) : $repo->findAll();

        return $this->json(
            data: $countries,
            status: Response::HTTP_OK,
            context: [AbstractNormalizer::GROUPS => ['service:read']]
        );
    }

    # GET /api/services/:id
    #[Route('/api/services/{id}', name: 'api.services.show', methods: ['GET'])]
    public function show(Service $service): JsonResponse
    {
        return $this->json(
            data: $service,
            status: Response::HTTP_OK,
            context: [AbstractNormalizer::GROUPS => ['service:read']]
        );
    }

    # POST /api/services
    #[Route('/api/services', name: 'api.services.create', methods: ['POST'])]  
    public function create(Request $request): JsonResponse
    {
        $data = $request->getContent();
        $service = $this->serializer->deserialize($data, Service::class, 'json');
        $this->em->persist($service);
        $this->em->flush();

        return $this->json(
            data: $service,
            status: Response::HTTP_CREATED,
            context: [AbstractNormalizer::GROUPS => ['service:read']]
        );
    }

    # PUT /api/services/:id
    #[Route('/api/services/{id}', name: 'api.services.update', methods: ['PUT'])]
    public function update(Request $request, Service $service): JsonResponse
    {
        if (!$service) return $this->json([]); # por si no existe

        $data = $request->getContent();
        $this->serializer->deserialize($data, Service::class, 'json', [AbstractNormalizer::OBJECT_TO_POPULATE => $service]);
        $this->em->flush();

        return $this->json(
            data: $service,
            status: Response::HTTP_OK,
            context: [AbstractNormalizer::GROUPS => ['service:read']]
        );
    }
}
