<?php

namespace App\Controller;

use App\Entity\User;
use App\Repository\UserRepository;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Validator\Validator\ValidatorInterface;
use Symfony\Component\PasswordHasher\Hasher\UserPasswordHasherInterface;
use Symfony\Component\Serializer\Normalizer\AbstractNormalizer;
use Symfony\Component\Serializer\SerializerInterface;

final class UserController extends AbstractController
{
    private $em;
    private $validator;
    private $serializer;
    private $passwordHasher;

    public function __construct(
        EntityManagerInterface $em, 
        ValidatorInterface $validator, 
        SerializerInterface $serializer,
        UserPasswordHasherInterface $passwordHasher
    )
    {
        $this->em = $em;
        $this->validator = $validator;
        $this->serializer = $serializer;
        $this->passwordHasher = $passwordHasher;
    }

    # Render template
    #[Route('/usuarios', name: 'user.index', methods:['GET'])]
    public function index(): Response
    {
        return $this->render('user/index.html.twig');
    }

    # GET /api/users
    #[Route('/api/users', name: 'api.users.list', methods: ['GET'])]
    public function list(Request $request, UserRepository $repo): JsonResponse
    {
        $nombre = $request->query->get('nombre');
        $users = ($nombre) ? $repo->findByNombre($nombre) : $repo->findAll();

        return $this->json(
            data: $users,
            status: Response::HTTP_OK,
            context: [AbstractNormalizer::GROUPS => ['user:read']]
        );
    }

    # GET /api/users/:id
    #[Route('/api/users/{id}', name: 'api.users.show', methods: ['GET'])]
    public function show(User $user): JsonResponse
    {
        return $this->json(
            data: $user,
            status: Response::HTTP_OK,
            context: [AbstractNormalizer::GROUPS => ['user:read']]
        );
    }

    # POST /api/users
    #[Route('/api/users', name: 'api.users.create', methods: ['POST'])]
    public function create(Request $request): JsonResponse
    {
        $data = $request->getContent();
        $user = $this->serializer->deserialize($data, User::class, 'json');
        $user->setPassword($this->passwordHasher->hashPassword($user, $user->getPassword()));

        $this->em->persist($user);
        $this->em->flush();

        return $this->json(
            data: $user,
            status: Response::HTTP_CREATED,
            context: [AbstractNormalizer::GROUPS => ['user:read']]
        );
    }

    # PUT /api/users/:id
    #[Route('/api/users/{id}', name: 'api.users.update', methods: ['PUT'])]
    public function update(Request $request, User $user): JsonResponse
    {
        if (!$user) return $this->json([]); # por si no existe el usuario

        $data = $request->getContent();
        $this->serializer->deserialize($data, User::class, 'json', [AbstractNormalizer::OBJECT_TO_POPULATE => $user]);
        $user->setPassword($this->passwordHasher->hashPassword($user, $user->getPassword()));
        $this->em->flush();

        return $this->json(
            data: $user,
            status: Response::HTTP_OK,
            context: [AbstractNormalizer::GROUPS => ['user:read']]
        );
    }

    # DELETE /api/users/:id
    #[Route('/api/users/{id}', name: 'api.users.delete', methods: ['DELETE'])]
    public function delete(User $user): JsonResponse
    {
        if (!$user) return $this->json([]); # por si no existe el usuario

        try {
            $this->em->remove($user);
            $this->em->flush();
            return $this->json(['status' => Response::HTTP_NO_CONTENT], Response::HTTP_NO_CONTENT);
        } catch (\Exception $e) {
            return $this->json([
                'status' => Response::HTTP_CONFLICT,
                'message' => 'No se puede eliminar el usuario: ' . $e->getMessage(),
            ], Response::HTTP_CONFLICT);
        }
    }
}
