<?php

namespace App\Controller;

use App\Entity\Provider;
use App\Repository\ProviderRepository;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Serializer\Normalizer\AbstractNormalizer;
use Symfony\Component\Serializer\SerializerInterface;
use Symfony\Component\Validator\Validator\ValidatorInterface;

final class ProviderController extends AbstractController
{
    private $em;
    private $validator;
    private $serializer;

    public function __construct(EntityManagerInterface $em, ValidatorInterface $validator, SerializerInterface $serializer)
    {
        $this->em = $em;
        $this->validator = $validator;
        $this->serializer = $serializer;
    }

    # Render template providers - proveedores
    #[Route('/proveedores', name: 'providers.index', methods:['GET'])]   
    public function index(): Response
    {
        return $this->render('provider/index.html.twig');
    }

    # GET /api/providers
    #[Route('/api/providers', name: 'api.providers.list', methods: ['GET'])]
    public function list(Request $request, ProviderRepository $repo): JsonResponse
    {
        $nombre = $request->query->get('nombre');
        $providers = ($nombre) ? $repo->findByNombre($nombre) : $repo->findAll();

        return $this->json(
            data: $providers,
            status: Response::HTTP_OK,
            context: [AbstractNormalizer::GROUPS => ['provider:read']] 
        );
    }

    # GET /api/providers/:id
    #[Route('/api/providers/{id}', name: 'api.providers.show', methods: ['GET'])]
    public function show(Provider $provider): JsonResponse
    {
        return $this->json(
            data: $provider,
            status: Response::HTTP_OK,
            context: [AbstractNormalizer::GROUPS => ['provider:read']]
        );
    }

    # POST /api/providers
    #[Route('/api/providers', name: 'api.providers.create', methods: ['POST'])]  
    public function create(Request $request): JsonResponse
    {
        $data = $request->getContent();
        $provider = $this->serializer->deserialize($data, Provider::class, 'json');
        $this->em->persist($provider);
        $this->em->flush();

        return $this->json(
            data: $provider,
            status: Response::HTTP_CREATED,
            context: [AbstractNormalizer::GROUPS => ['provider:read']]
        );
    }

    # PUT /api/providers/:id
    #[Route('/api/providers/{id}', name: 'api.providers.update', methods: ['PUT'])]
    public function update(Request $request, Provider $provider): JsonResponse
    {
        if (!$provider) return $this->json([]); # por si no existe

        $data = $request->getContent();
        $this->serializer->deserialize($data, Provider::class, 'json', [AbstractNormalizer::OBJECT_TO_POPULATE => $provider]);
        $this->em->flush();

        return $this->json(
            data: $provider,
            status: Response::HTTP_OK,
            context: [AbstractNormalizer::GROUPS => ['provider:read']]
        );
    }
}
